/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/***

Copyright (c) 1997  Hewlett-Packard

Module Name:
    IdentOs.c	(Identify Operating System ... at compile/run time)

Abstract:
    This file includes all functions necessary to determine the operating
    system "intended" at compile time and actually in use at run time.
    i.e. "Compiled For" is simply a question of defines but 
    "Run Time" returns the actual OS in use.

Notes:
    ** IMPORTANT **
    The "true" operating system is returned except if
    compiled for DOS (will always return DOS even if running in
    a window under NT etc.)

Environments:
    Unix, DOS, Win 3.x, Win 95/98, Win NT

Revision History:
    Original; SCRobinson, 6/97

***/

/* 
// Make the compiler defines a little less messy 
// AND (MOST IMPORTANTLY) take care of multiple defines.
*/

#if defined(UNIX)
#   define OS_IS_UNIX
#   include <stdio.h>

#elif (defined(_WIN32) || defined(_WIN64)) && !defined(_DOS)
#   define OS_IS_WIN32
#   pragma warning(disable: 4115)
#   include <windows.h>
#   pragma warning(default: 4115)

#   if defined(WIN95) && defined(WINNT)
#     error You CANNOT define both WIN95 AND WINNT
#   endif

#   if defined(WIN95)
#     pragma message("Compiling with WIN95(98) defined")
#     define OS_IS_WIN95
#   elif defined(WINNT)
#     pragma message("Compiling with WINNT defined")
#     define OS_IS_WINNT
#   else
#     error Unknown Win32 Operating System
#   endif

#elif defined(WINDOWS) && !defined(_DOS)
#   define OS_IS_WINDOWS
#   include <windows.h>

#elif defined(DOS) || defined(_DOS)
#   define OS_IS_DOS
#   include <dos.h>
#   include <stdio.h>

#else    /* whoops */
#   error Operating System not defined in IdentOs.c
#endif

#include "identos.h"

/*
// The "short" version of the function
*/
B_OS_VERSION EXPORT GetOsVersionShort()
{
    return GetOsVersion( NULL, NULL );
}

/*
// The one and only real function in this file
// **IMPORTANT** ... 
// pszVersion MUST point to "char szVersion[80];"
//
*/
B_OS_VERSION EXPORT GetOsVersion( char pszVersion[80], B_OS_VERSION * pCompiledFor )

/************************* UNIX ***************************/
#ifdef  OS_IS_UNIX
{
    if( pCompiledFor )
        *pCompiledFor = OS_UNIX;
    if( pszVersion )
        sprintf(pszVersion, "Unix");
    return OS_UNIX;
}
#endif

/************************* DOS ****************************/
#ifdef  OS_IS_DOS
{
  unsigned char nMajor;       /* MS-DOS major version */
  unsigned char nMinor;       /* MS-DOS minor version, revision */
  unsigned char nOEMNumber;   /* OEM serial number */
  
  if( pCompiledFor )
    *pCompiledFor = OS_DOS;
  
  if( pszVersion )
  {
    _asm     mov ax,3000h
      _asm     int 21h
      _asm     mov nMinor, ah
      _asm     mov nMajor, al
      _asm     mov nOEMNumber, bh
      
      sprintf(pszVersion, "MS-DOS %u.%u OEM Serial Number %u",
      nMajor, nMinor, nOEMNumber);
  }
  return OS_DOS;
}
#endif


/************************* WIN3X **************************/
#ifdef  OS_IS_WINDOWS
{
    DWORD dwVersion = GetVersion();

    if( pCompiledFor )
        *pCompiledFor = OS_WIN3X;

    if (dwVersion < 0x80000000) 
    {
        /* Windows NT */
        if( pszVersion )
            wsprintf (pszVersion, "Windows NT %u.%u (Build: %u)",
                        (DWORD)(LOBYTE(LOWORD(dwVersion))),
                        (DWORD)(HIBYTE(LOWORD(dwVersion))),
                        (DWORD)(HIWORD(dwVersion)));
        return OS_WINNT;
    }
    else if (LOBYTE(LOWORD(dwVersion))<4) 
    {
        /* Win32s */
        if( pszVersion )
            wsprintf (pszVersion, "Win32s %u.%u (Build: %u)",
                        (DWORD)(LOBYTE(LOWORD(dwVersion))),
                        (DWORD)(HIBYTE(LOWORD(dwVersion))),
                        (DWORD)(HIWORD(dwVersion) & ~0x8000));
        return OS_WIN3X;
    } 
    else 
    {
        /* Windows 95 or 98 */
        if( pszVersion )
            wsprintf (pszVersion, "Windows 95/98 %u.%u (Build: %u)",
                        (DWORD)(LOBYTE(LOWORD(dwVersion))),
                        (DWORD)(HIBYTE(LOWORD(dwVersion))),
                        (DWORD)(HIWORD(dwVersion) & ~0x8000));
        return OS_WIN95;
    }
}
#endif


/************************* WIN32 **************************/
#ifdef  OS_IS_WIN32
{
    OSVERSIONINFO osvi;

    if( pCompiledFor )
#   ifdef OS_IS_WIN95
        *pCompiledFor = OS_WIN95;
#   else 
        *pCompiledFor = OS_WINNT;
#   endif

    memset(&osvi, 0, sizeof(OSVERSIONINFO));
    osvi.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
    GetVersionEx (&osvi);

    if (osvi.dwPlatformId == VER_PLATFORM_WIN32s)
    {
        if( pszVersion )
            wsprintf (pszVersion, "Win32s %d.%d (Build %d)",
                        osvi.dwMajorVersion,
                        osvi.dwMinorVersion,
                        osvi.dwBuildNumber & 0xFFFF);
        return OS_WIN3X;
    }
    else if (osvi.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS)
    {
        if( pszVersion )
            wsprintf (pszVersion, "Windows 95/98 %d.%d (Build %d)",
                        osvi.dwMajorVersion,
                        osvi.dwMinorVersion,
                        osvi.dwBuildNumber & 0xFFFF);
        return OS_WIN95;
    }
    else if (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT)
    {
        if( pszVersion )
            wsprintf (pszVersion, "Windows NT %d.%d (Build %d)",
                        osvi.dwMajorVersion,
                        osvi.dwMinorVersion,
                        osvi.dwBuildNumber & 0xFFFF);
        return OS_WINNT;
    }
  else
  {
    return OS_UNKNOWN;
  }
}
#endif

